<?php
// /user/daily_classes.php — Student portal: classes + full media gallery

// $ALLOW_BLOCKED = true; // blocked hone par bhi is page ko dekh sakte hain (optional)
require_once __DIR__ . '/_user_auth.php';
require_once __DIR__ . '/_user_layout.php';

// -----------------------------
// 1) Find student's latest course & batch
// -----------------------------
$course_id = 0; $batch_id = 0; $course_name = '';

$q = mysqli_query($conn, "
  SELECT r.course_id, r.batch_id, c.name AS course
  FROM registrations r
  JOIN courses c ON c.id = r.course_id
  WHERE r.student_id = {$student_id}
  ORDER BY r.approved_at DESC
  LIMIT 1
");
if ($r = mysqli_fetch_assoc($q)) {
  $course_id   = (int)$r['course_id'];
  $batch_id    = (int)$r['batch_id'];
  $course_name = $r['course'];
}

// -----------------------------
// 2) Load classes visible to this student
//    - is_active=1
//    - course_id match
//    - batch_id = student's batch OR NULL (course-wide)
//    - visible_from <= NOW()
// -----------------------------
$classes = [];
$classIds = [];
$allDays = [];

$sql = "
  SELECT d.id, d.title, d.description, d.visible_from, d.created_at, d.visible_to,
         c.name AS course, b.name AS batch,
         DATE(d.visible_from) as class_date,
         DAYNAME(d.visible_from) as day_name
  FROM daily_classes d
  LEFT JOIN courses c ON c.id = d.course_id
  LEFT JOIN batches b ON b.id = d.batch_id
  WHERE d.is_active = 1
    AND d.course_id = {$course_id}
    AND (d.batch_id = {$batch_id} OR d.batch_id IS NULL)
    AND d.visible_from <= NOW()
  ORDER BY d.visible_from DESC, d.created_at DESC
  LIMIT 100
";
$res = mysqli_query($conn, $sql);
while ($row = mysqli_fetch_assoc($res)) {
  $classes[] = $row;
  $classIds[] = (int)$row['id'];
  
  // Collect unique days for filter
  $dateKey = $row['class_date'];
  if (!isset($allDays[$dateKey])) {
    $allDays[$dateKey] = [
      'date' => $dateKey,
      'display' => date('d M, Y', strtotime($dateKey)),
      'day_name' => $row['day_name'],
      'count' => 0
    ];
  }
  $allDays[$dateKey]['count']++;
}

// Sort days in descending order (newest first)
krsort($allDays);

// Get selected day from query parameter
$selectedDay = $_GET['day'] ?? 'all';
if ($selectedDay !== 'all' && !isset($allDays[$selectedDay])) {
  $selectedDay = 'all';
}

// Filter classes by selected day
$filteredClasses = [];
if ($selectedDay === 'all') {
  $filteredClasses = $classes;
} else {
  foreach ($classes as $class) {
    if ($class['class_date'] === $selectedDay) {
      $filteredClasses[] = $class;
    }
  }
}

// -----------------------------
// 3) Preload ALL media for these classes (single query)
// -----------------------------
$mediaMap = []; // class_id => [file_path, ...]
if (!empty($classIds)) {
  $idList = implode(',', $classIds);
  $mq = mysqli_query($conn, "
    SELECT class_id, file_path
    FROM daily_class_media
    WHERE class_id IN ({$idList})
    ORDER BY id ASC
  ");
  while ($m = mysqli_fetch_assoc($mq)) {
    $cid = (int)$m['class_id'];
    if (!isset($mediaMap[$cid])) $mediaMap[$cid] = [];
    $mediaMap[$cid][] = $m['file_path'];
  }
}

// -----------------------------
// 4) Helper: make correct URL for stored files
//    Stored path is usually 'uploads/classes/xyz.ext'
//    Student pages are in /user => need '../' prefix
// -----------------------------
function media_url($relativePath) {
  $fp = ltrim($relativePath ?? '', '/');
  if ($fp === '') return '#';

  // Try ../uploads/... (since this file is in /user)
  $try1_fs = __DIR__ . '/../' . $fp;
  if (is_file($try1_fs)) return '../' . $fp;

  // Try /user/uploads/... (rare, but just in case)
  $try2_fs = __DIR__ . '/' . $fp;
  if (is_file($try2_fs)) return $fp;

  // Fallback (web server might still resolve)
  return (strpos($fp, 'uploads/') === 0) ? '../' . $fp : $fp;
}

// safe html
if (!function_exists('h')) {
  function h($s){ return htmlspecialchars($s ?? '', ENT_QUOTES, 'UTF-8'); }
}

user_header('Daily Classes');
?>

<style>
  :root {
    --primary: #4f46e5;
    --primary-light: #6366f1;
    --primary-dark: #4338ca;
    --secondary: #64748b;
    --success: #10b981;
    --warning: #f59e0b;
    --danger: #ef4444;
    --light: #f8fafc;
    --dark: #0f172a;
    --gray-100: #f1f5f9;
    --gray-200: #e2e8f0;
    --gray-300: #cbd5e1;
    --gray-400: #94a3b8;
    --gray-500: #64748b;
    --gray-600: #475569;
    --gray-700: #334155;
    --gray-800: #1e293b;
    --gray-900: #0f172a;
    --border-radius: 12px;
    --shadow-sm: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
    --shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
    --transition: all 0.3s ease;
  }

  body {
    background: linear-gradient(135deg, #f5f7fa 0%, #e4edf5 100%);
    min-height: 100vh;
    font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    color: var(--gray-800);
    line-height: 1.6;
  }

  .page-header {
    background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
    color: white;
    padding: 1.5rem;
    border-radius: var(--border-radius);
    margin-bottom: 1.5rem;
    box-shadow: var(--shadow-lg);
    position: relative;
    overflow: hidden;
  }

  .page-header::before {
    content: '';
    position: absolute;
    top: 0;
    right: 0;
    width: 200px;
    height: 100%;
    background: rgba(255, 255, 255, 0.1);
    transform: skewX(-20deg);
  }

  .page-title {
    font-size: 1.75rem;
    font-weight: 700;
    margin: 0 0 0.5rem 0;
    display: flex;
    align-items: center;
    gap: 0.75rem;
  }

  .page-subtitle {
    font-size: 1rem;
    opacity: 0.9;
    margin: 0;
  }

  /* Day Filter Styles */
  .day-filter-container {
    background: white;
    border-radius: var(--border-radius);
    padding: 1.25rem;
    margin-bottom: 1.5rem;
    box-shadow: var(--shadow);
    border: 1px solid var(--gray-200);
  }

  .filter-title {
    font-size: 1.125rem;
    font-weight: 600;
    margin: 0 0 1rem 0;
    color: var(--gray-800);
    display: flex;
    align-items: center;
    gap: 0.5rem;
  }

  .day-filters {
    display: flex;
    gap: 0.75rem;
    flex-wrap: wrap;
  }

  .day-filter-btn {
    padding: 0.75rem 1.25rem;
    border-radius: var(--border-radius);
    border: 2px solid var(--gray-200);
    background: white;
    color: var(--gray-700);
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition);
    display: flex;
    align-items: center;
    gap: 0.5rem;
    text-decoration: none;
  }

  .day-filter-btn:hover {
    border-color: var(--primary-light);
    color: var(--primary);
    transform: translateY(-1px);
  }

  .day-filter-btn.active {
    background: var(--primary);
    border-color: var(--primary);
    color: white;
  }

  .day-badge {
    background: var(--gray-100);
    color: var(--gray-600);
    padding: 0.25rem 0.5rem;
    border-radius: 999px;
    font-size: 0.75rem;
    font-weight: 600;
  }

  .day-filter-btn.active .day-badge {
    background: rgba(255, 255, 255, 0.2);
    color: white;
  }

  .class-card {
    background: white;
    border-radius: var(--border-radius);
    padding: 1.5rem;
    box-shadow: var(--shadow);
    margin-bottom: 1.5rem;
    border: 1px solid var(--gray-200);
    transition: var(--transition);
    position: relative;
    overflow: hidden;
  }

  .class-card:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
  }

  .class-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 4px;
    height: 100%;
    background: var(--primary);
  }

  .class-head {
    display: flex;
    gap: 1rem;
    align-items: flex-start;
    justify-content: space-between;
    margin-bottom: 1rem;
    flex-wrap: wrap;
  }

  .class-title {
    font-weight: 700;
    color: var(--dark);
    font-size: 1.25rem;
    margin: 0 0 0.5rem 0;
    line-height: 1.4;
  }

  .class-meta {
    color: var(--gray-500);
    font-size: 0.875rem;
    display: flex;
    flex-wrap: wrap;
    gap: 0.5rem;
    align-items: center;
  }

  .chips {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
  }

  .chip {
    padding: 0.25rem 0.75rem;
    border-radius: 999px;
    border: 1px solid var(--gray-200);
    font-size: 0.75rem;
    background: var(--gray-100);
    color: var(--gray-700);
    font-weight: 500;
    display: flex;
    align-items: center;
    gap: 0.25rem;
  }

  .chip.primary {
    background: var(--primary);
    color: white;
    border-color: var(--primary);
  }

  .chip.success {
    background: var(--success);
    color: white;
    border-color: var(--success);
  }

  .chip.warning {
    background: var(--warning);
    color: white;
    border-color: var(--warning);
  }

  .desc {
    color: var(--gray-700);
    white-space: pre-wrap;
    margin: 1rem 0;
    line-height: 1.6;
    padding: 1rem;
    background: var(--gray-50);
    border-radius: 8px;
    border-left: 3px solid var(--primary-light);
  }

  .gallery {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
    gap: 1rem;
    margin-top: 1.5rem;
  }

  .file-box {
    border: 1px solid var(--gray-200);
    border-radius: 10px;
    overflow: hidden;
    background: white;
    transition: var(--transition);
    box-shadow: var(--shadow-sm);
  }

  .file-box:hover {
    transform: translateY(-3px);
    box-shadow: var(--shadow);
  }

  .file-box img {
    display: block;
    width: 100%;
    height: 180px;
    object-fit: cover;
  }

  .file-meta {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 0.75rem;
    font-size: 0.875rem;
    color: var(--gray-600);
    background: var(--gray-50);
  }

  .file-meta a {
    font-weight: 600;
    color: var(--primary);
    text-decoration: none;
    display: flex;
    align-items: center;
    gap: 0.25rem;
  }

  .file-meta a:hover {
    color: var(--primary-dark);
  }

  .video-wrap video {
    width: 100%;
    height: 180px;
    display: block;
    border-bottom: 1px solid var(--gray-200);
  }

  .btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    padding: 0.5rem 1rem;
    background: var(--primary);
    color: white;
    border-radius: 6px;
    font-weight: 500;
    font-size: 0.875rem;
    text-decoration: none;
    border: none;
    cursor: pointer;
    transition: var(--transition);
    gap: 0.5rem;
  }

  .btn:hover {
    background: var(--primary-dark);
    color: white;
    transform: translateY(-1px);
  }

  .btn-outline {
    background: transparent;
    color: var(--primary);
    border: 1px solid var(--primary);
  }

  .btn-outline:hover {
    background: var(--primary);
    color: white;
  }

  .empty-state {
    text-align: center;
    padding: 3rem 1rem;
    color: var(--gray-500);
    background: white;
    border-radius: var(--border-radius);
    box-shadow: var(--shadow);
  }

  .empty-state-icon {
    font-size: 3rem;
    margin-bottom: 1rem;
    opacity: 0.5;
  }

  .empty-state h3 {
    font-size: 1.25rem;
    margin: 0 0 0.5rem 0;
    color: var(--gray-600);
  }

  .empty-state p {
    margin: 0;
    max-width: 400px;
    margin: 0 auto;
  }

  .security-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: transparent;
    z-index: 9999;
    pointer-events: none;
  }

  .no-select {
    user-select: none;
    -webkit-user-select: none;
    -moz-user-select: none;
    -ms-user-select: none;
  }

  .no-context {
    -webkit-touch-callout: none;
    -webkit-user-select: none;
    -khtml-user-select: none;
    -moz-user-select: none;
    -ms-user-select: none;
    user-select: none;
  }

  .results-count {
    color: var(--gray-600);
    font-size: 0.875rem;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
  }

  @media (max-width: 768px) {
    .class-head {
      flex-direction: column;
      align-items: flex-start;
      gap: 0.75rem;
    }
    
    .chips {
      width: 100%;
    }
    
    .gallery {
      grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
    }
    
    .day-filters {
      flex-direction: column;
    }
    
    .day-filter-btn {
      justify-content: space-between;
    }
  }

  @media (max-width: 480px) {
    .gallery {
      grid-template-columns: 1fr;
    }
    
    .page-title {
      font-size: 1.5rem;
    }
    
    .class-card {
      padding: 1rem;
    }
  }
</style>

<!-- Security overlay to prevent interaction -->
<div class="security-overlay no-select no-context"></div>

<div class="page-header">
  <h1 class="page-title">
    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
      <path d="M12 14V16M8 10V8C8 5.79086 9.79086 4 12 4C14.2091 4 16 5.79086 16 8V10M5 10H19C20.1046 10 21 10.8954 21 12V18C21 19.1046 20.1046 20 19 20H5C3.89543 20 3 19.1046 3 18V12C3 10.8954 3.89543 10 5 10Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
    </svg>
    Your Daily Classes
  </h1>
  <p class="page-subtitle">
    <?= h($course_name) ?> • Batch: <?= $batch_id ? h($batch_id) : 'All Batches' ?>
  </p>
</div>

<!-- Day Filter Section -->
<div class="day-filter-container">
  <h3 class="filter-title">
    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
      <path d="M3 4C3 3.44772 3.44772 3 4 3H20C20.5523 3 21 3.44772 21 4V6.58579C21 6.851 20.8946 7.10536 20.7071 7.29289L14.2929 13.7071C14.1054 13.8946 14 14.149 14 14.4142V17L10 21V14.4142C10 14.149 9.89464 13.8946 9.70711 13.7071L3.29289 7.29289C3.10536 7.10536 3 6.851 3 6.58579V4Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
    </svg>
    Filter by Day
  </h3>
  
  <div class="day-filters">
    <!-- All Days Button -->
    <a href="?day=all" class="day-filter-btn <?= $selectedDay === 'all' ? 'active' : '' ?>">
      <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M9 12H15M9 16H15M17 21H7C5.89543 21 5 20.1046 5 19V7C5 5.89543 5.89543 5 7 5H17C18.1046 5 19 5.89543 19 7V19C19 20.1046 18.1046 21 17 21Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
      </svg>
      All Classes
      <span class="day-badge"><?= count($classes) ?></span>
    </a>
    
    <!-- Individual Day Buttons -->
    <?php foreach ($allDays as $day): ?>
      <a href="?day=<?= h($day['date']) ?>" class="day-filter-btn <?= $selectedDay === $day['date'] ? 'active' : '' ?>">
        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
          <path d="M8 7V3M16 7V3M7 11H17M5 21H19C20.1046 21 21 20.1046 21 19V7C21 5.89543 20.1046 5 19 5H5C3.89543 5 3 5.89543 3 7V19C3 20.1046 3.89543 21 5 21Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
        </svg>
        <?= h($day['day_name']) ?>, <?= h($day['display']) ?>
        <span class="day-badge"><?= $day['count'] ?></span>
      </a>
    <?php endforeach; ?>
  </div>
</div>

<!-- Results Count -->
<div class="results-count">
  <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
    <path d="M9 12H15M9 16H15M17 21H7C5.89543 21 5 20.1046 5 19V7C5 5.89543 5.89543 5 7 5H17C18.1046 5 19 5.89543 19 7V19C19 20.1046 18.1046 21 17 21Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
  </svg>
  Showing <?= count($filteredClasses) ?> class<?= count($filteredClasses) !== 1 ? 'es' : '' ?>
  <?php if ($selectedDay !== 'all'): ?>
    for <?= date('l, d M Y', strtotime($selectedDay)) ?>
  <?php endif; ?>
</div>

<?php if (empty($filteredClasses)): ?>
  <div class="empty-state">
    <div class="empty-state-icon">📚</div>
    <h3>No classes found</h3>
    <p>
      <?php if ($selectedDay === 'all'): ?>
        No classes have been posted yet. Please check back later.
      <?php else: ?>
        No classes found for <?= date('l, d M Y', strtotime($selectedDay)) ?>. Try selecting a different day.
      <?php endif; ?>
    </p>
  </div>
<?php endif; ?>

<?php foreach ($filteredClasses as $row):
  $cid   = (int)$row['id'];
  $media = $mediaMap[$cid] ?? [];
?>
  <div class="class-card no-select no-context">
    <div class="class-head">
      <div style="flex: 1;">
        <h2 class="class-title"><?= h($row['title']) ?></h2>
        <div class="class-meta">
          <span><?= h($row['course']) ?></span>
          <span>•</span>
          <span><?= $row['batch'] ? h($row['batch']) : 'All Batches' ?></span>
          <span>•</span>
          <span>Posted: <?= date('d M Y, H:i', strtotime($row['created_at'])) ?></span>
        </div>
      </div>
      <div class="chips">
        <span class="chip primary">
          <svg width="12" height="12" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
            <path d="M12 22C17.5228 22 22 17.5228 22 12C22 6.47715 17.5228 2 12 2C6.47715 2 2 6.47715 2 12C2 17.5228 6.47715 22 12 22Z" stroke="currentColor" stroke-width="2"/>
            <path d="M12 6V12L16 14" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
          </svg>
          Visible: <?= date('d M, H:i', strtotime($row['visible_from'])) ?>
        </span>
        <?php if ($row['visible_to']): ?>
          <span class="chip warning">
            <svg width="12" height="12" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M12 9V14M12 17H12.01M21 12C21 16.9706 16.9706 21 12 21C7.02944 21 3 16.9706 3 12C3 7.02944 7.02944 3 12 3C16.9706 3 21 7.02944 21 12Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
            </svg>
            Expires: <?= h($row['visible_to']) ?>
          </span>
        <?php endif; ?>
      </div>
    </div>

    <?php if (!empty($row['description'])): ?>
      <div class="desc"><?= h($row['description']) ?></div>
    <?php endif; ?>

    <?php if (!empty($media)): ?>
      <h3 style="font-size: 1rem; margin: 1.5rem 0 0.75rem 0; color: var(--gray-700); display: flex; align-items: center; gap: 0.5rem;">
        <svg width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
          <path d="M4 16L8 12L4 8M12 8H20M12 16H20" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
        </svg>
        Attached Media (<?= count($media) ?>)
      </h3>
      <div class="gallery">
        <?php foreach ($media as $fp):
          $url = media_url($fp);
          $ext = strtolower(pathinfo($fp, PATHINFO_EXTENSION));
          if (in_array($ext, ['jpg','jpeg','png','gif','webp'])): ?>
            <!-- Image -->
            <div class="file-box">
              <a href="<?= $url ?>" target="_blank" title="Open image">
                <img src="<?= $url ?>" alt="Class image">
              </a>
              <div class="file-meta">
                <span>Image</span>
                <a href="<?= $url ?>" target="_blank">
                  <svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M18 13V19C18 19.5304 17.7893 20.0391 17.4142 20.4142C17.0391 20.7893 16.5304 21 16 21H5C4.46957 21 3.96086 20.7893 3.58579 20.4142C3.21071 20.0391 3 19.5304 3 19V8C3 7.46957 3.21071 6.96086 3.58579 6.58579C3.96086 6.21071 4.46957 6 5 6H11" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                    <path d="M15 3H21V9" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                    <path d="M10 14L21 3" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                  </svg>
                  Open
                </a>
              </div>
            </div>
          <?php elseif (in_array($ext, ['mp4','webm','ogg'])): ?>
            <!-- Video -->
            <div class="file-box">
              <div class="video-wrap">
                <video src="<?= $url ?>" controls preload="metadata" controlsList="nodownload"></video>
              </div>
              <div class="file-meta">
                <span>Video</span>
                <a href="<?= $url ?>" target="_blank">
                  <svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M18 13V19C18 19.5304 17.7893 20.0391 17.4142 20.4142C17.0391 20.7893 16.5304 21 16 21H5C4.46957 21 3.96086 20.7893 3.58579 20.4142C3.21071 20.0391 3 19.5304 3 19V8C3 7.46957 3.21071 6.96086 3.58579 6.58579C3.96086 6.21071 4.46957 6 5 6H11" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                    <path d="M15 3H21V9" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                    <path d="M10 14L21 3" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                  </svg>
                  Open
                </a>
              </div>
            </div>
          <?php elseif ($ext === 'pdf'): ?>
            <!-- PDF -->
            <div class="file-box" style="display:flex;flex-direction:column;align-items:center;justify-content:center;height:180px;text-align:center;">
              <div style="font-size:48px;margin-bottom:10px;">📄</div>
              <div style="font-size:14px;font-weight:500;margin-bottom:8px;">PDF Document</div>
              <a class="btn" href="<?= $url ?>" target="_blank" style="margin-top:8px;">
                <svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M12 10V16M12 16L9 13M12 16L15 13M6 20H18C18.5304 20 19.0391 19.7893 19.4142 19.4142C19.7893 19.0391 20 18.5304 20 18V6C20 5.46957 19.7893 4.96086 19.4142 4.58579C19.0391 4.21071 18.5304 4 18 4H6C5.46957 4 4.96086 4.21071 4.58579 4.58579C4.21071 4.96086 4 5.46957 4 6V18C4 18.5304 4.21071 19.0391 4.58579 19.4142C4.96086 19.7893 5.46957 20 6 20Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                </svg>
                View PDF
              </a>
            </div>
          <?php else: ?>
            <!-- Other -->
            <div class="file-box" style="display:flex;flex-direction:column;align-items:center;justify-content:center;height:180px;text-align:center;">
              <div style="font-size:48px;margin-bottom:10px;">📁</div>
              <div style="font-size:14px;font-weight:500;margin-bottom:8px;"><?= h(strtoupper($ext)) ?> File</div>
              <a class="btn" href="<?= $url ?>" target="_blank" style="margin-top:8px;">
                <svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M21 15V19C21 19.5304 20.7893 20.0391 20.4142 20.4142C20.0391 20.7893 19.5304 21 19 21H5C4.46957 21 3.96086 20.7893 3.58579 20.4142C3.21071 20.0391 3 19.5304 3 19V15" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                  <path d="M7 10L12 15L17 10" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                  <path d="M12 15V3" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                </svg>
                Download
              </a>
            </div>
          <?php endif; ?>
        <?php endforeach; ?>
      </div>
    <?php else: ?>
      <div class="muted" style="margin-top:1rem;padding:1rem;background:var(--gray-50);border-radius:8px;text-align:center;color:var(--gray-500);">
        No media attached to this class.
      </div>
    <?php endif; ?>
  </div>
<?php endforeach; ?>

<script>
// Security measures to prevent screenshots, screen recording, and right-click
document.addEventListener('DOMContentLoaded', function() {
  // Prevent right-click context menu
  document.addEventListener('contextmenu', function(e) {
    e.preventDefault();
    return false;
  });
  
  // Prevent text selection
  document.addEventListener('selectstart', function(e) {
    e.preventDefault();
    return false;
  });
  
  // Prevent drag and drop
  document.addEventListener('dragstart', function(e) {
    e.preventDefault();
    return false;
  });
  
  // Disable keyboard shortcuts for screenshots (not fully reliable but helps)
  document.addEventListener('keydown', function(e) {
    // Disable Print Screen key
    if (e.key === 'PrintScreen') {
      e.preventDefault();
      return false;
    }
    
    // Disable Ctrl+Shift+I (DevTools), Ctrl+Shift+C (Inspect Element), etc.
    if ((e.ctrlKey && e.shiftKey && (e.key === 'I' || e.key === 'C' || e.key === 'J')) || 
        (e.key === 'F12')) {
      e.preventDefault();
      return false;
    }
  });
  
  // Add a watermark overlay to discourage screenshots
  const watermark = document.createElement('div');
  watermark.style.position = 'fixed';
  watermark.style.top = '0';
  watermark.style.left = '0';
  watermark.style.width = '100%';
  watermark.style.height = '100%';
  watermark.style.pointerEvents = 'none';
  watermark.style.zIndex = '9998';
  watermark.style.background = 'transparent';
  watermark.style.opacity = '0.03';
  watermark.style.fontSize = '120px';
  watermark.style.fontWeight = 'bold';
  watermark.style.color = '#000';
  watermark.style.display = 'flex';
  watermark.style.alignItems = 'center';
  watermark.style.justifyContent = 'center';
  watermark.style.transform = 'rotate(-30deg)';
  watermark.innerText = 'CONFIDENTIAL';
  document.body.appendChild(watermark);
  
  // Disable video right-click and download
  document.querySelectorAll('video').forEach(video => {
    video.addEventListener('contextmenu', e => e.preventDefault());
    video.controlsList = 'nodownload';
  });
  
  // Disable image right-click and drag
  document.querySelectorAll('img').forEach(img => {
    img.addEventListener('contextmenu', e => e.preventDefault());
    img.setAttribute('draggable', 'false');
  });
  
  // Log attempts to take screenshots (limited capability in browser)
  window.addEventListener('blur', function() {
    // This might indicate the user switched to another app like a screenshot tool
    console.warn('Page lost focus - possible screenshot attempt');
  });
});
</script>

<?php user_footer(); ?>